//=============================================================================
// PRTN_AbsCore.js - Last modified 2021/05/07
//=============================================================================

/*:
* @plugindesc ARPG用の機能を実装するセット
* @target MZ
* @author Perutan（けんせい）
* @url http://twitter.com/kensei1203
*
* @param 非ダメージ時アニメーションID
* @desc ダメージが通らなかったときに表示するアニメーションID
* @type animation
* @default 16
*
* @param ヒット時アニメーションID
* @desc 攻撃が通ったときに表示するアニメーションID
* @type animation
* @default 6
*
* @param 強ヒット時アニメーションID
* @desc クリティカルや弱点攻撃が通ったときに表示するアニメーションID
* @type animation
* @default 6
*
* @param 攻撃ダメージ値用変数ID
* @desc 算出されたダメージを格納する変数のID
* @type variable
* @default 98
*
* @param 被ダメージ値用変数ID
* @desc 算出されたダメージを格納する変数のID
* @type variable
* @default 10
*
* @param 被ダメージスイッチID
* @desc 被ダメージ時にONにするスイッチのID
* @type switch
* @default 84
*
* @param 攻撃属性引き渡し用変数ID
* @desc 攻撃時の属性を格納する変数のID
* @type variable
* @default 69
*
* @param 対象者イベントID用変数ID
* @desc 攻撃対象のイベントIDを格納する変数のID
* @type variable
* @default 7
*
* @param 攻撃者イベントID用変数ID
* @desc 被ダメージ対象のイベントIDを格納する変数のID
* @type variable
* @default 10
*
* @param ザコダメージコモンイベントID
* @desc ザコ敵に攻撃が当たったときに呼び出すコモンイベントのID
* @type common_event
* @default 0
*
* @param ザコクリティカルダメージコモンイベントID
* @desc ザコ敵にクリティカル攻撃が当たったときに呼び出すコモンイベントのID
* @type common_event
* @default 0
*
* @param ボスダメージコモンイベントID
* @desc ボス敵に攻撃が当たったときに呼び出すコモンイベントのID
* @type common_event
* @default 0
*
* @param 攻撃判定無効用スイッチID
* @desc 攻撃判定を無効にするためのスイッチのID
* @type switch
* @default 9
*
* @param ランクアップ変数ID
* @desc 難易度を調整するためのレートを格納する変数のID（内容は％）
* @type variable
* @default 0
*
* @param ヒット数カウント変数ID
* @desc 一度の攻撃がヒットした格納する変数のID
* @type variable
* @default 0
*
* @param 大型敵の横サイズ変数ID
* @desc <ABSbigger>で指定される大型敵の横サイズ（拡張分）を格納する変数のID
* @type variable
* @default 0
*
* @param 大型敵の縦サイズ変数ID
* @desc <ABSbigger>で指定される大型敵の縦サイズ（拡張分）を格納する変数のID
* @type variable
* @default 0
*
* @param 敵撃破時オンにするセルフスイッチ
* @desc 敵撃破時オンにするセルフスイッチ
* @type string
* @default B
*
*
* @help
*
* 【機能１】
* 攻撃時の当たり判定を発生させるプラグインコマンド。
* 判定内に、<ABSenemy:N>タグが定義されているイベントが
* 存在するとヒット処理をおこないます。
*
* Plugin Command:
*   GetCollision n1 n2 n3 n4 n5 n6
*   n1 = [X方向の範囲]
*   n2 = [Y方向の範囲]
*   n3 = [相対中心位置X]
*   n4 = [相対中心位置Y]
*   n5 = [スキルID（省略で通常攻撃）]
*   n6 = [主体] 省略でプレイヤーキャラ
*
* Plugin Command:
*   StartDealDamage
*   ダメージ処理をおこないます。
*   「GetCollision」で取得されたイベントID、攻撃スキル、ダメージ値が受け渡されます。
*
* Plugin Command:
*   StartSufferDamage
*   敵からアクターに与えられるダメージ処理をおこないます。
*
*
* 【機能２】
* イベントとプレイヤーの距離を計算して返します。
*
*  this.eventDist(イベントID)
*
*
* 【機能３】
* 指定したイベントに紐付けられたエネミーの各種パラメータを取得します。
*
*  this.eventName(イベントID)  //名前
*  this.eventHp(イベントID)  //最大HP
*  this.eventMp(イベントID)  //最大MP
*  this.eventAtk(イベントID) //攻撃力
*  this.eventDef(イベントID) //防御力
*  this.eventMat(イベントID) //魔法力
*  this.eventMdf(イベントID) //魔法防御
*  this.eventAgi(イベントID) //俊敏性
*  this.eventLuk(イベントID) //運
*  this.eventExp(イベントID) //経験値
*  this.eventGold(イベントID)  //所持金
*
*
* 【機能４】
*  画面上に敵キャラのHPゲージを表示します。
*
*
* Thnaks to
*  フェルミウム湾 様
*  サンシロ 様
*  kotonoha 様
*  暴兎 様
*
*
*
*/

(function () {
  'use strict'

  //=============================================================================
  // プラグインパラメータの取得
  //=============================================================================
  var pluginName = 'PRTN_AbsCore';
  var parameters = PluginManager.parameters('PRTN_AbsCore');
  var noDamageAnimId = Number(parameters['非ダメージ時アニメーションID']);
  var hitDamageAnimId = Number(parameters['ヒット時アニメーションID']);
  var criDamageAnimId = Number(parameters['強ヒット時アニメーションID']);
  var dealDamageVarId = Number(parameters['攻撃ダメージ値用変数ID']);
  var sufferDamageVarId = Number(parameters['被ダメージ値用変数ID']);
  var sufferDamageSwiId = Number(parameters['被ダメージスイッチID']);
  var targetEidVarId = Number(parameters['対象者イベントID用変数ID']);
  var attackerEidVarId = Number(parameters['攻撃者イベントID用変数ID']);
  var zCommonEventId = Number(parameters['ザコダメージコモンイベントID']);
  var zCritCommonEventId = Number(parameters['ザコクリティカルダメージコモンイベントID']);
  var bCommonEventId = Number(parameters['ボスダメージコモンイベントID']);
  var noHitSwiId = Number(parameters['攻撃判定無効用スイッチID']);
  var attributeVarId = Number(parameters['攻撃属性引き渡し用変数ID']);
  var rankRateVarId = Number(parameters['ランクアップ変数ID']);
  var hitCountVarId = Number(parameters['ヒット数カウント変数ID']);
  var biggerSizeXVarId = Number(parameters['大型敵の横サイズ変数ID']);
  var biggerSizeYVarId = Number(parameters['大型敵の縦サイズ変数ID']);
  var selfSwitchByDefeat = String(parameters['敵撃破時オンにするセルフスイッチ']);


  //=============================================================================
  // Game_CharacterBase
  //=============================================================================
  var _Game_CharacterBase_initMembers = Game_CharacterBase.prototype.initMembers;
  Game_CharacterBase.prototype.initMembers = function () {
    _Game_CharacterBase_initMembers.apply(this, arguments);
    this.ABSweakened = false;
    this.ABSinvincible = false;
    this.ABShp = 0;
    this.ABSignoreTouchEnemy = false;
    this.ABSdefeated = false;
    this.ABSshotCharge = 0;
  };

  //=============================================================================
  // Game_Event
  //=============================================================================
  var _Game_Event_initMembers = Game_Event.prototype.initMembers;
  Game_Event.prototype.initMembers = function () {
    _Game_Event_initMembers.apply(this, arguments);
    this.ABSnam = 0;
    this.ABSmhp = 0;
    this.ABSmmp = 0;
    this.ABSatk = 0;
    this.ABSdef = 0;
    this.ABSmat = 0;
    this.ABSmdf = 0;
    this.ABSagi = 0;
    this.ABSluk = 0;
    this.ABSexp = 0;
    this.ABSgld = 0;
    this.ABSTargetId = 0;
    this.ABSslowCounter = 0;
    this.ABSfire = false;
    //this._battler = new Game_Enemy(this.event().meta.enemy);
  };

  var _Game_Event_initialize = Game_Event.prototype.initialize;
  Game_Event.prototype.initialize = function (mapId, eventId) {
    _Game_Event_initialize.apply(this, arguments);
    this._battler = this.event().meta.enemy > 0 ? new Game_Enemy(this.event().meta.enemy) : null;
  };

  //=============================================================================
  // Game_Interpreter
  //=============================================================================

  // プラグインコマンドの定義

  // for MZ
  /*
  const script = document.currentScript;
  const param = PluginManagerEx.createParameter(script);

  PluginManagerEx.registerCommand(script, 'GetCollision', function(args) {
    var argsMV = [args.width, args.height, args.offsetX, args.offsetY, args.skillId, args.indiv, args.magnification];
    this.getCollision(argsMV);
  });

  PluginManagerEx.registerCommand(script, 'StartDealDamage', function(args) {
    var argsMV = [args.targetId, args.indiv, args.magnification];
    this.startDealDamage(argsMV);
  });

  PluginManagerEx.registerCommand(script, 'StartSufferDamage', function(args) {
    this.startSufferDamage(args);
  });
  */

  // for MV
  var _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
  Game_Interpreter.prototype.pluginCommand = function (command, args) {
    _Game_Interpreter_pluginCommand.call(this, command, args);
    //console.log(command+' '+args);

    // 攻撃判定　コマンド定義
    if (command === 'GetCollision') {
      this.getCollision(args);
    };

    // 攻撃ダメージ処理　コマンド定義
    if (command === 'StartDealDamage') {
      this.startDealDamage(args);
    };

    // 被ダメージ処理　コマンド定義
    if (command === 'StartSufferDamage') {
      this.startSufferDamage(args);
    };
  };

  //=============================================================================
  // マップ敵のセルフスイッチとライフを初期化
  Game_Interpreter.prototype.setupPrBattleParameter = function () {
    for (var eventID = 1; eventID <= 999; eventID++) {
      var mapID = this._mapId;
      if (!$dataMap.events[eventID]) continue;

      if ($gameMap.event(eventID).event().meta.ABSbreakable && !$gameMap.event(eventID).event().meta.ABSatOnce) {
        $gameSelfSwitches.setValue([mapID, eventID, "A"], false);
        $gameSelfSwitches.setValue([mapID, eventID, "B"], false);
        $gameSelfSwitches.setValue([mapID, eventID, "C"], false);
        $gameSelfSwitches.setValue([mapID, eventID, "D"], false);
        $gameMap.event(eventID).ABShp = 0;
        $gameMap.event(eventID).ABSdefeated = false;
      }
    }
  };

  Game_Interpreter.prototype.setupPrBattleParameterPart2 = function (eventID) {
    console.log('EV: ' + eventID);
    if (eventID) {
      if (Number($gameMap.event(eventID).event().meta.ABSenemy) > 0) {
        $gameMap.event(eventID).ABSnam = this.eventName(eventID);
        $gameMap.event(eventID).ABSmhp = this.eventHp(eventID);
        $gameMap.event(eventID).ABSmmp = this.eventMp(eventID);
        $gameMap.event(eventID).ABSatk = this.eventAtk(eventID);
        $gameMap.event(eventID).ABSdef = this.eventDef(eventID);
        $gameMap.event(eventID).ABSmat = this.eventMat(eventID);
        $gameMap.event(eventID).ABSmdf = this.eventMdf(eventID);
        $gameMap.event(eventID).ABSagi = this.eventAgi(eventID);
        $gameMap.event(eventID).ABSluk = this.eventLuk(eventID);
        $gameMap.event(eventID).ABSexp = this.eventExp(eventID);
        $gameMap.event(eventID).ABSgld = this.eventGold(eventID);
      }
    }
  };

  //=============================================================================
  // 攻撃判定
  Game_Interpreter.prototype.getCollision = function (args) {
    var collidWidth = Number(args[0]);
    var collidHeight = Number(args[1]);
    var shiftX = Number(args[2]);
    var shiftY = Number(args[3]);
    var attribute = Number(args[4] || 1);
    var indiv = Number(args[5] || -1);
    var partyMember = Number(args[6] || 1);
    var magnification = Number(args[7] || 1);

    if (indiv === 0) {
      indiv = this.eventId();
    }

    if ($gameSwitches.value(noHitSwiId)) return;

    $gameVariables._data[hitCountVarId] = 0;

    // 通常サイズの敵
    var targetId = $gameMap.eventIdsInDistance(collidWidth, collidHeight, shiftX, shiftY, indiv);
    for (var i = 0; i < targetId.length; i++) {
      var target = targetId[i];
      var targetCharacter = $gameMap.event(target);
      if (!targetCharacter.event().meta.ABSbigger && !targetCharacter.event().meta.ABSweed) {
        $gameVariables._data[targetEidVarId] = target;
        $gameVariables._data[attributeVarId] = attribute;
        this.startDealDamage(target, indiv, magnification, partyMember);
        if (!targetCharacter.event().meta.ABSweed) {
          //console.log('TARGET : ' + target);
          this._interpreter = new Game_Interpreter();
          targetCharacter._interpreter.setup($dataCommonEvents[zCommonEventId].list, target);
        }
        if (targetCharacter.ABShp >= this.eventHp(target)) {
          var key = [$gameMap.mapId(), target, selfSwitchByDefeat];
          $gameSelfSwitches._data[key] = true;
          $gameMap.event(key[1]).refresh();
          targetCharacter.ABSdefeated = true;
        }
      }

      if (target != 0 && targetCharacter.event().meta.ABSweed) {
        var key = [$gameMap.mapId(), target, selfSwitchByDefeat];
        $gameSelfSwitches._data[key] = true;
        $gameMap.event(key[1]).refresh();
        targetCharacter.ABSdefeated = true;
      }
    }

    // 大きい敵
    var biggerExtenderX = $gameVariables.value(biggerSizeXVarId);
    var biggerExtenderY = $gameVariables.value(biggerSizeYVarId);
    var biggerTargetId = $gameMap.eventIdsInDistance(collidWidth + biggerExtenderX, collidHeight + biggerExtenderY, shiftX, shiftY + biggerExtenderY / 2, indiv);
    for (var i = 0; i < biggerTargetId.length; i++) {
      var target = biggerTargetId[i];
      var targetCharacter = $gameMap.event(target);
      if (targetCharacter.event().meta.ABSbigger) {
        $gameVariables._data[targetEidVarId] = target;
        $gameVariables._data[attributeVarId] = attribute;
        this.startDealDamage(target, indiv, magnification, partyMember);
        if (targetCharacter.event().meta.ABScallCommon) {
          this._interpreter = new Game_Interpreter();
          targetCharacter._interpreter.setup($dataCommonEvents[bCommonEventId].list, target);
        }
        if (targetCharacter.ABShp >= this.eventHp(target)) {
          var key = [$gameMap.mapId(), target, selfSwitchByDefeat];
          $gameSelfSwitches._data[key] = true;
          $gameMap.event(key[1]).refresh();
          targetCharacter.ABSdefeated = true;
        }
      }
    }
  };

  //=============================================================================
  // 攻撃ダメージ処理
  Game_Interpreter.prototype.startDealDamage = function (targetId, indiv, magnification, partyMember) {
    var target = this.character(targetId)._eventId;

    var attribute = $gameVariables._data[attributeVarId]
    var damageResult = this.arpgMakeDamegeValue(target, attribute, partyMember);
    var dealDamage = damageResult[0];
    var criSuccess = damageResult[1];
    var hitWeakPoint = damageResult[2];

    if ($gameMap.event(target).ABSweakened === true) {
      dealDamage *= 2;
    }
    $gameMap.event(target).ABShp += dealDamage;
    $gameVariables._data[dealDamageVarId] = dealDamage;

    var mhp = this.eventHp(target);
    var hp = mhp - $gameMap.event(target).ABShp;
    if (hp < 0) hp = 0;
    $gameVariables._data[hitCountVarId] += 1;

    if (Imported.MKR_EventGauge && !$gameSwitches.value(80)) {
      var enemyEv = $gameMap.event(target);
      $gameMap.setGaugeMaxValue(enemyEv._eventId, mhp);
      $gameMap.setGaugeValue(enemyEv._eventId, hp);
      $gameMap.showGaugeWindow(enemyEv._eventId);
      if (hp / mhp < 0.25) {
        $gameMap.setGaugeColor1(enemyEv._eventId, 2);
        $gameMap.setGaugeColor2(enemyEv._eventId, 2);
      }
    }

    if (hp / mhp < 0.25) {
      var ev = $gameMap.event(target);
      //ev.setTone(64, 0, 12, 24);
      ev._tone = [80, 10, 0, 100];
    }

    if (hitDamageAnimId > 0) {
      if (dealDamage > 0) {

        var targetWeight = this.eventMdf(target);
        if (targetWeight < 100 && hp > 0 && $gameMap.event(target)._directionFix === false) {
          if ($gameMap.event(target).ABSdef > 0) magnification *= 0.25;
        }
        if (hitWeakPoint || criSuccess) {
          $gameTemp.requestAnimation([$gameMap.event(target)], criDamageAnimId);
        } else {
          $gameTemp.requestAnimation([$gameMap.event(target)], hitDamageAnimId);
        }
      }
    }
    if (noDamageAnimId > 0) {
      if (dealDamage <= 0) {
        $gameTemp.requestAnimation([$gameMap.event(target)], noDamageAnimId);
      }
    }
  };


  // 被ダメージ処理　処理内容
  Game_Interpreter.prototype.startSufferDamage = function (args) {
    var eId = Number(args[0]);
    var attribute = Number(args[1]);
    var partyMember = Number(args[2]);
    //var attribute = $gameVariables._data[attributeVarId];
    var sufferDamage = this.arpgSufferDamegeValue(eId, attribute, partyMember);
    $gameVariables._data[sufferDamageVarId] = sufferDamage[0];
    $gameSwitches.setValue(sufferDamageSwiId, true)
  };

  //=============================================================================
  // 攻撃ダメージ計算用の関数を定義
  Game_Interpreter.prototype.arpgMakeDamegeValue = function (eId, attribute, partyMember) {
    if (eId === undefined) {
      eId = this.eventId();
    }
    if (partyMember === undefined) {
      partyMember = 1;
    }
    if (attribute === undefined) {
      var atkSkillId = 1; //通常攻撃
    } else {
      var atkSkillId = attribute; //特殊攻撃 = スキルID
    }

    var followersDamageRate = 0.8;
    var actor = $gameActors.actor(partyMember);
    var memberActorIds = $gameParty.members().map((obj) => {
      return obj._actorId;
    });
    var order = memberActorIds.indexOf(actor._actorId);
    var enemyId = Number($gameMap.event(eId).event().meta.ABSenemy);
    var enemy = $dataEnemies[enemyId];

    console.log(actor, enemy, atkSkillId);
    var damageResult = this.arpgCalcDamage(actor, enemy, atkSkillId, eId);
    if (order > 0) damageResult[0] = Math.floor(damageResult[0] * followersDamageRate);
    //console.log(damageResult);
    return damageResult;
  };

  // 被ダメージ計算用の関数を定義
  Game_Interpreter.prototype.arpgSufferDamegeValue = function (eId, attribute, partyMember) {
    if (eId === undefined) {
      eId = this.eventId();
    }
    if (partyMember === undefined) {
      partyMember = 1;
    }
    if (attribute === undefined) {
      var atkSkillId = 1; //通常攻撃
    } else {
      var atkSkillId = attribute; //特殊攻撃 = スキルID
    }

    var followersDamageRate = 0.8;
    var actor = $gameActors.actor(partyMember);
    var memberActorIds = $gameParty.members().map((obj) => {
      return obj._actorId;
    });
    var order = memberActorIds.indexOf(actor._actorId);
    var enemyId = Number($gameMap.event(eId).event().meta.ABSenemy);
    var enemy = $dataEnemies[enemyId];

    console.log(enemy, actor, atkSkillId);
    var damageResult = this.arpgCalcDamage(enemy, actor, atkSkillId, eId);
    if (order > 0) damageResult[0] = Math.floor(damageResult[0] * followersDamageRate);
    return damageResult;
  };

  // ダメージ計算
  Game_Interpreter.prototype.arpgCalcDamage = function (attacker, blocker, atkSkillId, eId) {
    if (!attacker.hasOwnProperty('params')) {
      var a = attacker;
    } else {
      var a = {
        "mhp": attacker.params[0] + $gameMap.event(eId).ABSmhp,
        "mmp": attacker.params[1] + $gameMap.event(eId).ABSmmp,
        "atk": attacker.params[2] + $gameMap.event(eId).ABSatk,
        "def": attacker.params[3] + $gameMap.event(eId).ABSdef,
        "mat": attacker.params[4] + $gameMap.event(eId).ABSmat,
        "mdf": attacker.params[5] + $gameMap.event(eId).ABSmdf,
        "agi": attacker.params[6] + $gameMap.event(eId).ABSagi,
        "luk": attacker.params[7] + $gameMap.event(eId).ABSluk,
      };
      //var v = $gameVariables._data;  
    }
    console.log("a.atk :"+a.atk);

    if (!blocker.hasOwnProperty('params')) {
      var b = blocker;
    } else {
      var b = {
        "mhp": blocker.params[0] + $gameMap.event(eId).ABSmhp,
        "mmp": blocker.params[1] + $gameMap.event(eId).ABSmmp,
        "atk": blocker.params[2] + $gameMap.event(eId).ABSatk,
        "def": blocker.params[3] + $gameMap.event(eId).ABSdef,
        "mat": blocker.params[4] + $gameMap.event(eId).ABSmat,
        "mdf": blocker.params[5] + $gameMap.event(eId).ABSmdf,
        "agi": blocker.params[6] + $gameMap.event(eId).ABSagi,
        "luk": blocker.params[7] + $gameMap.event(eId).ABSluk,
      };
      //var v = $gameVariables._data;  
    }

    //スキルの各種情報を取得
    var n = atkSkillId;
    var atkCritical = $dataSkills[n].damage.critical; //boolean
    var atkElementId = $dataSkills[n].damage.elementId;
    var atkFormula = $dataSkills[n].damage.formula;
    var atkType = $dataSkills[n].damage.type;
    var atkVariance = $dataSkills[n].damage.variance;
    var atkMpCost = $dataSkills[n].mpCost;
    var atkName = $dataSkills[n].name;
    var atkTpCost = $dataSkills[n].tpCost;
    var atkTpGain = $dataSkills[n].tpGain;
    var atkCriticalRate = !attacker.hasOwnProperty('params') ? a.xparam(2) : 0;

    //通常攻撃の場合、装備中の武器の属性を適用
    if (!attacker.hasOwnProperty('params') && atkElementId === -1) {
      var equipWpnId = a._equips[0]._itemId;
      if (equipWpnId > 0) {
        for (var i = 0; i < $dataWeapons[equipWpnId].traits.length; i++) {
          if ($dataWeapons[equipWpnId].traits[i].code === 31) {
            atkElementId = $dataWeapons[equipWpnId].traits[i].dataId;
          }
        }  
      }
    }

    //計算
    var result = Math.ceil(eval(atkFormula));

    var hitWeakPoint = false;
    //防御側の属性有効度チェック
    if (blocker.hasOwnProperty('params')) {
      var defElementsRateArr = blocker.traits;
      for (var i = 0; i < defElementsRateArr.length; i++) {
        if (defElementsRateArr[i].code === 11) {
          var defElementsRateId = defElementsRateArr[i].dataId;
          if (defElementsRateId === atkElementId) {
            result = result * defElementsRateArr[i].value;
            if (defElementsRateArr[i].value > 1) {
              hitWeakPoint = true;
            }
          }
        }
      }  
    }

    //分散度適用
    var amp = Math.floor(Math.max(Math.abs(result) * atkVariance / 100, 0));
    var v = Math.randomInt(amp + 1) + Math.randomInt(amp + 1) - amp;
    result = result >= 0 ? result + v : result - v;

    //クリティカル判定
    if (atkCritical) {
      var criJudge = Math.floor(Math.random() * 100);
      var criSuccess = criJudge < atkCriticalRate * 100 ? true : false;
      if (criSuccess) {
        result *= 3;
      }
    }

    var damageResult = [result > 0 ? Math.round(result) : 0, criSuccess, hitWeakPoint];

    $gameVariables._data[dealDamageVarId] = damageResult[0];
    /*
    if (damageResult[0] === 0) {
    } else {
      var cpdArgs = new Array(
        String(eId), String($gameVariables._data[dealDamageVarId])
      );
      if (criSuccess || hitWeakPoint) {
        this.pluginCommand('CPD_MP_DAMAGE', cpdArgs);
      } else {
        this.pluginCommand('CPD_DAMAGE', cpdArgs);
      }
    }
    */

    return damageResult;
  };

  //=============================================================================
  // 指定したイベントに紐付けられたエネミーの各種パラメータを取得する関数を定義
  //=============================================================================

  //------------------------------------------------------------------------------
  // 名前
  Game_Interpreter.prototype.eventName = function (eId) {
    if (eId === undefined) {
      eId = this.eventId();
    }
    if (!$gameMap.event(eId)) return;
    return $dataEnemies[Number($gameMap.event(eId).event().meta.ABSenemy)].name || 0;
  };

  //------------------------------------------------------------------------------
  // 最大HP
  Game_Interpreter.prototype.eventHp = function (eId) {
    if (eId === undefined) {
      eId = this.eventId();
    }
    if (!$gameMap.event(eId)) return;
    var rankRate = 100;
    if ($dataEnemies[Number($gameMap.event(eId).event().meta.ABSenemy)].meta.ABSrankUp) {
      var rankRate = rankRateVarId > 0 ? $gameVariables.value(rankRateVarId) : 100;
    }
    return Math.round($dataEnemies[Number($gameMap.event(eId).event().meta.ABSenemy)].params[0] * ((rankRate * (rankRate / 100)) / 100)) || 0;
  };

  //------------------------------------------------------------------------------
  // 最大MP
  Game_Interpreter.prototype.eventMp = function (eId) {
    if (eId === undefined) {
      eId = this.eventId();
    }
    if (!$gameMap.event(eId)) return;
    var rankRate = 100;
    if ($dataEnemies[Number($gameMap.event(eId).event().meta.ABSenemy)].meta.ABSrankUp) {
      var rankRate = rankRateVarId > 0 ? $gameVariables.value(rankRateVarId) : 100;
    }
    return Math.round($dataEnemies[Number($gameMap.event(eId).event().meta.ABSenemy)].params[1] * rankRate / 100) || 0;
  };

  //------------------------------------------------------------------------------
  // 攻撃力
  Game_Interpreter.prototype.eventAtk = function (eId) {
    if (eId === undefined) {
      eId = this.eventId();
    }
    if (!$gameMap.event(eId)) return;
    var rankRate = 100;
    if ($dataEnemies[Number($gameMap.event(eId).event().meta.ABSenemy)].meta.ABSrankUp) {
      var rankRate = rankRateVarId > 0 ? $gameVariables.value(rankRateVarId) : 100;
    }
    return Math.round($dataEnemies[Number($gameMap.event(eId).event().meta.ABSenemy)].params[2] * rankRate / 100) || 0;
  };

  //------------------------------------------------------------------------------
  // 防御力
  Game_Interpreter.prototype.eventDef = function (eId) {
    if (eId === undefined) {
      eId = this.eventId();
    }
    if (!$gameMap.event(eId)) return;
    var rankRate = 100;
    if ($dataEnemies[Number($gameMap.event(eId).event().meta.ABSenemy)].meta.ABSrankUp) {
      var rankRate = rankRateVarId > 0 ? $gameVariables.value(rankRateVarId) : 100;
    }
    //
    rankRate = 100;
    //
    return Math.round($dataEnemies[Number($gameMap.event(eId).event().meta.ABSenemy)].params[3] * rankRate / 100) || 0;
  };

  //------------------------------------------------------------------------------
  // 魔法力
  Game_Interpreter.prototype.eventMat = function (eId) {
    if (eId === undefined) {
      eId = this.eventId();
    }
    if (!$gameMap.event(eId)) return;
    var rankRate = 100;
    if ($dataEnemies[Number($gameMap.event(eId).event().meta.ABSenemy)].meta.ABSrankUp) {
      var rankRate = rankRateVarId > 0 ? $gameVariables.value(rankRateVarId) : 100;
    }
    return Math.round($dataEnemies[Number($gameMap.event(eId).event().meta.ABSenemy)].params[4] * rankRate / 100) || 0;
  };

  //------------------------------------------------------------------------------
  // 魔法防御
  Game_Interpreter.prototype.eventMdf = function (eId) {
    if (eId === undefined) {
      eId = this.eventId();
    }
    if (!$gameMap.event(eId)) return;
    var rankRate = 100;
    if ($dataEnemies[Number($gameMap.event(eId).event().meta.ABSenemy)].meta.ABSrankUp) {
      var rankRate = rankRateVarId > 0 ? $gameVariables.value(rankRateVarId) : 100;
    }
    //
    rankRate = 100;
    //    
    return Math.round($dataEnemies[Number($gameMap.event(eId).event().meta.ABSenemy)].params[5] * rankRate / 100) || 0;
  };

  //------------------------------------------------------------------------------
  // 俊敏性
  Game_Interpreter.prototype.eventAgi = function (eId) {
    if (eId === undefined) {
      eId = this.eventId();
    }
    if (!$gameMap.event(eId)) return;
    var rankRate = 100;
    if ($dataEnemies[Number($gameMap.event(eId).event().meta.ABSenemy)].meta.ABSrankUp) {
      var rankRate = rankRateVarId > 0 ? $gameVariables.value(rankRateVarId) : 100;
    }
    return Math.round($dataEnemies[Number($gameMap.event(eId).event().meta.ABSenemy)].params[6] * rankRate / 100) || 0;
  };

  //------------------------------------------------------------------------------
  // 運
  Game_Interpreter.prototype.eventLuk = function (eId) {
    if (eId === undefined) {
      eId = this.eventId();
    }
    if (!$gameMap.event(eId)) return;
    var rankRate = 100;
    if ($dataEnemies[Number($gameMap.event(eId).event().meta.ABSenemy)].meta.ABSrankUp) {
      var rankRate = rankRateVarId > 0 ? $gameVariables.value(rankRateVarId) : 100;
    }
    return Math.round($dataEnemies[Number($gameMap.event(eId).event().meta.ABSenemy)].params[7] * rankRate / 100) || 0;
  };

  //------------------------------------------------------------------------------
  // 経験値
  Game_Interpreter.prototype.eventExp = function (eId) {
    if (eId === undefined) {
      eId = this.eventId();
    }
    if (!$gameMap.event(eId)) return;
    var rankRate = 100;
    if ($dataEnemies[Number($gameMap.event(eId).event().meta.ABSenemy)].meta.ABSrankUp) {
      var rankRate = rankRateVarId > 0 ? $gameVariables.value(rankRateVarId) : 100;
    }
    return Math.round($dataEnemies[Number($gameMap.event(eId).event().meta.ABSenemy)].exp * rankRate / 100) || 0;
  };

  //------------------------------------------------------------------------------
  // 所持金
  Game_Interpreter.prototype.eventGold = function (eId) {
    if (eId === undefined) {
      eId = this.eventId();
    }
    if (!$gameMap.event(eId)) return;
    var rankRate = 100;
    if ($dataEnemies[Number($gameMap.event(eId).event().meta.ABSenemy)].meta.ABSrankUp) {
      var rankRate = rankRateVarId > 0 ? $gameVariables.value(rankRateVarId) : 100;
    }
    return Math.round($dataEnemies[Number($gameMap.event(eId).event().meta.ABSenemy)].gold) || 0;
  };

  //=============================================================================
  // イベントとプレイヤーの距離を取得する関数を定義
  //=============================================================================

  Game_Interpreter.prototype.eventDist = function (eId, directionShift) {
    if (eId === undefined) {
      eId = this.eventId();
    }
    if (directionShift === undefined) {
      directionShift = 0;
    }

    var direction = this.character(eId)._direction;
    switch (direction) {
      case 2:
        var shiftX = 0;
        var shiftY = directionShift;
        break;
      case 4:
        var shiftX = directionShift * -1;
        var shiftY = 0;
        break;
      case 6:
        var shiftX = directionShift;
        var shiftY = 0;
        break;
      case 8:
        var shiftX = 0;
        var shiftY = directionShift * -1;
        break;
    }

    var realDeltaX = this.character(-1)._realX - (this.character(eId)._realX + shiftX);
    var realDeltaY = this.character(-1)._realY - (this.character(eId)._realY + shiftY);
    return Math.sqrt(Math.pow(realDeltaX, 2) + Math.pow(realDeltaY, 2));
  };

  //=============================================================================
  // イベント同士の距離を取得する関数を定義
  //=============================================================================

  Game_Interpreter.prototype.eventToEventDist = function (eId1, eId2) {
    if (eId1 === undefined) {
      eId1 = this.eventId();
    }
    if (eId2 === undefined) {
      eId2 = this.eventId();
    }
    var realDeltaX = this.character(eId1)._realX - this.character(eId2)._realX;
    var realDeltaY = this.character(eId1)._realY - this.character(eId2)._realY;
    return Math.sqrt(Math.pow(realDeltaX, 2) + Math.pow(realDeltaY, 2));
  };

  //=============================================================================
  // 自動振り向き用の関数
  //=============================================================================

  // プレイヤーから引数距離以内のイベントリスト
  Game_Map.prototype.autoLockEnemiesInDistance = function (distance) {
    return this.events().filter(
      function (event) {
        var subj = $gamePlayer;
        var deltaX = event._realX - subj._realX;
        var deltaY = event._realY - subj._realY;
        var delta = Math.sqrt(Math.pow(deltaX, 2) + Math.pow(deltaY, 2));
        return delta <= distance
          && event.event().meta.ABSenemy > 0
          && event.event().meta.ABSbreakable === true
          && !event.event().meta.ABSweed
          && event.ABSdefeated === false
          && event.ABSinvincible === false
      }
    );
  };

  // プレイヤーから引数距離以内のイベントIDリスト
  Game_Map.prototype.autoLockEnemyIdInDistance = function (distance) {
    var events = this.autoLockEnemiesInDistance(distance);
    if (events.length <= 0) return 0;
    var deltaList = events.map(
      function (event) {
        var subj = $gamePlayer;
        var deltaX = event._realX - subj._realX;
        var deltaY = event._realY - subj._realY;
        var delta = Math.sqrt(Math.pow(deltaX, 2) + Math.pow(deltaY, 2));
        return delta;
      }
    );
    var elmNum = deltaList.indexOf(Math.min.apply(null, deltaList));
    var eventId = events[elmNum].eventId();
    //console.log(eventId);
    return eventId;
  };

  // プレイヤーから引数距離以内のイベントID
  // 存在しない場合は 0 を返す
  /*
  Game_Map.prototype.autoLockEnemyIdInDistance = function (distance) {
    var eventIds = this.autoLockEnemyIdsInDistance(distance);
    var eventId = eventIds.length > 0 ? eventIds[0] : 0;
    return eventId;
  };
  */

  //=============================================================================
  // 接触判定用の関数
  //=============================================================================

  var _Scene_Map_update = Scene_Map.prototype.update;
  Scene_Map.prototype.update = function () {
    _Scene_Map_update.call(this);
    //$gameMap.touchEnemy(1.1);
  };

  Game_Map.prototype.touchEnemy = function (dist) {
    var collideEnemy = $gameMap.ABSenemyIdInDistance(dist);
    if (collideEnemy > 0) {
      var attribute = $gameVariables._data[attributeVarId];
      var sufferDamage = Game_Interpreter.prototype.arpgSufferDamegeValue(collideEnemy);
      $gameVariables._data[sufferDamageVarId] = sufferDamage;
      $gameSwitches.setValue(sufferDamageSwiId, true);
    }
  };

  // プレイヤーから引数距離以内のイベントリスト
  Game_Map.prototype.enemiesInDistance = function (distance, indiv) {
    return this.events().filter(
      function (event) {
        var subj = indiv === undefined ? $gamePlayer : $gameMap.event(indiv);
        var deltaX = event._realX - subj._realX;
        var deltaY = event._realY - subj._realY;
        var delta = Math.sqrt(Math.pow(deltaX, 2) + Math.pow(deltaY, 2));
        return delta <= distance
          && event.event().meta.ABSenemy > 0
          && !event.event().meta.ABSoneShot
          && event.ABSignoreTouchEnemy === false
          && event.ABSdefeated === false
      }
    );
  };

  // プレイヤーから引数距離以内のイベントIDリスト
  Game_Map.prototype.ABSenemyIdsInDistance = function (distance, indiv) {
    var events = this.enemiesInDistance(distance, indiv);
    var eventIds = events.map(
      function (event) {
        return event.eventId();
      }
    );
    return eventIds;
  };

  // プレイヤーから引数距離以内のイベントID
  // 存在しない場合は 0 を返す
  Game_Map.prototype.ABSenemyIdInDistance = function (distance, indiv) {
    var eventIds = this.ABSenemyIdsInDistance(distance, indiv);
    var eventId = eventIds.length > 0 ? eventIds[0] : 0;
    return eventId;
  };

  //=============================================================================
  // 攻撃距離測定関数を定義 Supported by サンシロ
  //=============================================================================

  // プレイヤーから引数距離以内のイベントリスト
  Game_Map.prototype.eventsInDistance = function (width, height, offsetX, offsetY, indiv) {
    return this.events().filter(
      function (event) {
        if (indiv === -1) {
          var originX = $gamePlayer._realX + offsetX;
          var originY = $gamePlayer._realY + offsetY;
        } else {
          var originX = $gameMap.event(indiv)._realX + offsetX;
          var originY = $gameMap.event(indiv)._realY + offsetY;
        }
        var hitX = 0;
        var hitY = 0;
        if (originX - width / 2 <= event._realX && event._realX <= originX + width / 2) {
          hitX = 1;
        }
        if (originY - height / 2 <= event._realY && event._realY <= originY + height / 2) {
          hitY = 1;
        }
        return hitX + hitY === 2
          && event.ABSinvincible === false
          && event.event().meta.ABSbreakable
          && event.ABSdefeated === false
        //&& Math.sqrt(Math.pow(($gamePlayer.prJumpHeight - event.prJumpHeight),2)) < 4
      }
    );
  };

  // プレイヤーから引数距離以内のイベントIDリスト
  Game_Map.prototype.eventIdsInDistance = function (width, height, offsetX, offsetY, indiv) {
    var events = this.eventsInDistance(width, height, offsetX, offsetY, indiv);
    var eventIds = events.map(
      function (event) {
        return event.eventId();
      }
    );
    return eventIds;
  };

  // プレイヤーから引数距離以内のイベントID
  // 存在しない場合は 0 を返す
  Game_Map.prototype.eventIdInDistance = function (distance) {
    var eventIds = this.eventIdsInDistance(distance * 2, distance * 2, 0, 0, -1);
    var eventId = eventIds.length > 0 ? eventIds[0] : 0;
    return eventId;
  };

  // プレイヤーから引数距離以内のすり抜けなしイベントリスト
  Game_Map.prototype.eventsInDistanceNt = function (distance) {
    var events = this.eventsInDistance(distance);
    events = events.filter(
      function (event) {
        return !event.isThrough();
      }
    );
    return events;
  };

  // プレイヤーから引数距離以内のすり抜けなしイベントIDリスト
  Game_Map.prototype.eventIdsInDistanceNt = function (distance) {
    var events = this.eventsInDistanceNt(distance);
    var eventIds = events.map(
      function (event) {
        return event.eventId();
      }
    );
    return eventIds;
  };

  // プレイヤーから引数距離以内のすり抜けなしイベントID
  // 存在しない場合は 0 を返す
  Game_Map.prototype.eventIdInDistanceNt = function (distance) {
    var eventIds = this.eventIdsInDistanceNt(distance);
    var eventId = eventIds.length > 0 ? eventIds[0] : 0;
    return eventId;
  };



})();
